$(document).ready(function () {
    'use strict';

    const $form = $('.needs-validation');
    const $steps = $('.step');
    const $indicators = $('.step-indicator');
    const $connectors = $('.step-connector');
    let currentStep = 0;
    let completedSteps = new Set(); // Track which steps have been completed

    const showStep = (index) => {
        $steps.removeClass('active').eq(index).addClass('active');
        updateIndicators();
    };

    const updateIndicators = () => {
        $indicators.each(function (i) {
            const $indicator = $(this);

            // Remove all state classes first
            $indicator.removeClass('active completed completing');

            if (completedSteps.has(i)) {
                // Step is completed - show green tick
                $indicator.addClass('completed');

                // Update connector line
                $connectors.eq(i).addClass('completed');
            } else if (i === currentStep) {
                // Current active step
                $indicator.addClass('active');
            }
        });
    };

    const validateStep = (stepIndex) => {
        const $currentStepEl = $steps.eq(stepIndex);
        const $inputs = $currentStepEl.find('input, select, textarea');
        let isValid = true;

        $inputs.each(function () {
            const $input = $(this);

            // Check validity of each input
            if (!this.checkValidity()) {
                isValid = false;
            }

            // Add Bootstrap validation classes for visual feedback
            $input.addClass('was-validated');

            // For radio buttons, handle validation display differently
            if ($input.attr('type') === 'radio') {
                const name = $input.attr('name');
                const $radioGroup = $currentStepEl.find(`input[name="${name}"]`);
                const isRadioGroupValid = $radioGroup.is(':checked');

                $radioGroup.each(function () {
                    $(this).addClass('was-validated');
                    if (!isRadioGroupValid) {
                        this.setCustomValidity('Please select an option');
                    } else {
                        this.setCustomValidity('');
                    }
                });

                if (!isRadioGroupValid) {
                    isValid = false;
                }
            }
        });

        // Add was-validated to the current step container
        $currentStepEl.addClass('was-validated');

        return isValid;
    };

    const checkStepCompletion = (stepIndex) => {
        const $stepEl = $steps.eq(stepIndex);
        const $inputs = $stepEl.find('input, select, textarea');
        let isComplete = true;

        $inputs.each(function () {
            const $input = $(this);
            const type = $input.attr('type');

            if (type === 'radio') {
                // For radio buttons, check if any in the group is selected
                const name = $input.attr('name');
                const $radioGroup = $stepEl.find(`input[name="${name}"]`);
                const hasSelection = $radioGroup.is(':checked');
                if (!hasSelection) {
                    isComplete = false;
                }
            } else if (type === 'checkbox' && $input.prop('required')) {
                // For required checkboxes
                if (!$input.is(':checked')) {
                    isComplete = false;
                }
            } else if ($input.prop('required') && !$input.val().trim()) {
                // For other required inputs
                isComplete = false;
            }
        });

        return isComplete;
    };

    const markStepAsCompleted = (stepIndex) => {
        const $indicator = $indicators.eq(stepIndex);

        // Add completing animation class
        $indicator.addClass('completing');

        // Add to completed steps after animation delay
        setTimeout(() => {
            completedSteps.add(stepIndex);
            updateIndicators();
        }, 300);
    };

    const validateEntireForm = () => {
        let isValid = true;

        // Validate all steps
        for (let i = 0; i < $steps.length; i++) {
            if (!validateStep(i)) {
                isValid = false;
            }
        }

        return isValid;
    };

    const clearValidationClasses = (stepIndex) => {
        const $stepEl = $steps.eq(stepIndex);
        $stepEl.removeClass('was-validated');
        $stepEl.find('input, select, textarea').removeClass('was-validated').each(function() {
            this.setCustomValidity('');
        });
    };

    // Next step button event - KEEPS VALIDATION
    $(document).on('click', '.next-step', function () {
        // Validate current step only
        if (validateStep(currentStep)) {
            // Mark current step as completed with animation
            markStepAsCompleted(currentStep);

            // Move to next step
            currentStep++;
            if (currentStep >= $steps.length) {
                currentStep = $steps.length - 1;
            }
            showStep(currentStep);
        }
    });

    // Previous step button event
    $(document).on('click', '.prev-step', function () {
        currentStep--;
        if (currentStep < 0) {
            currentStep = 0;
        }
        showStep(currentStep);
    });

    // Form submit event
    $form.on('submit', function (e) {
        e.preventDefault();
        e.stopPropagation();

        // Validate entire form on submit
        if (validateEntireForm()) {
            // Mark final step as completed
            markStepAsCompleted(currentStep);

            setTimeout(() => {
                // Show success message with better styling
                $('body').append(`
      <div class="modal fade" id="successModal" tabindex="-1">
        <div class="modal-dialog modal-dialog-centered">
          <div class="modal-content border-0 shadow-lg">
            <div class="modal-body text-center p-5">
              <div class="mb-4">
                <i class="fas fa-check-circle text-success" style="font-size: 4rem;"></i>
              </div>
              <h3 class="text-success mb-3">Registration Successful!</h3>
              <p class="text-muted mb-4">Your form has been submitted successfully. Welcome aboard!</p>
              <button type="button" class="btn btn-modern btn-success-modern" data-bs-dismiss="modal">
                <i class="fas fa-thumbs-up me-2"></i>Awesome!
              </button>
            </div>
          </div>
        </div>
      </div>
    `);
                $('#successModal').modal('show');
            }, 600);
        } else {
            // If form is invalid, show the first step with errors
            for (let i = 0; i < $steps.length; i++) {
                const $invalidInputs = $steps.eq(i).find('input:invalid, select:invalid, textarea:invalid');
                if ($invalidInputs.length > 0) {
                    currentStep = i;
                    showStep(currentStep);
                    break;
                }
            }
        }
    });

    // Remove validation classes and check completion when user interacts
    $form.on('input change', 'input, select, textarea', function () {
        const $target = $(this);

        // Remove validation classes
        $target.removeClass('was-validated');

        // For radio buttons, remove validation from all in the group
        if ($target.attr('type') === 'radio') {
            const name = $target.attr('name');
            const $radioGroup = $form.find(`input[name="${name}"]`);
            $radioGroup.removeClass('was-validated').each(function () {
                this.setCustomValidity('');
            });
        }

        // Check if current step is now complete or incomplete
        const $stepContainer = $target.closest('.step');
        const stepIndex = $steps.index($stepContainer);

        if (stepIndex !== -1) {
            if (checkStepCompletion(stepIndex)) {
                if (!completedSteps.has(stepIndex) && stepIndex !== currentStep) {
                    // Step became complete
                    completedSteps.add(stepIndex);
                    updateIndicators();
                }
            } else {
                if (completedSteps.has(stepIndex)) {
                    // Step became incomplete
                    completedSteps.delete(stepIndex);
                    updateIndicators();
                }
            }
        }
    });

    // Step indicator click navigation - REMOVED
    // Headers are no longer clickable - navigation only through Next/Previous buttons

    // Initialize - check which steps are already complete on page load
    $steps.each(function (index) {
        if (checkStepCompletion(index) && index !== currentStep) {
            completedSteps.add(index);
        }
    });

    updateIndicators();

    // Add smooth scrolling for mobile
    $(window).on('resize', function () {
        if ($(window).width() <= 768) {
            $('.step.active').get(0)?.scrollIntoView({ behavior: 'smooth' });
        }
    });
});